/*
   This program creates a pipeline between itself and its child.
   This program then lets you "watch" how the buffering of the
   parent's output and the child's input interact.

   This program prints to stderr a continuous count of how many
   bytes have been sent down the pipeline to the child. Whenever
   that count stalls, that means that one of the buffers between
   the parent and child is full, so the parent has blocked. By
   watching this value, seeing when it changes and by how much
   it changes, you can determine how the buffering is working.
*/
import java.io.*;

public class Ex2_WatchBuffering
{
   public static void main(String[] args) throws IOException
   {
      // Command line for running the child process.
      String[] cmd = {"java", "Ex2_WatchBuffering_Child"};
      // Execute the command line.
      final Process process = Runtime.getRuntime().exec(cmd);
      // Give this parent the ability to shut down the child.
      Runtime.getRuntime().addShutdownHook(new Thread(){public void run(){process.destroy();}});


      // A stream that writes data to the child's stdin stream (the "pipe").
      PrintStream stdinOfChild =  new PrintStream( process.getOutputStream() );

      // Write 10,000 bytes to the child's input buffer.
      for (int counter = 1; counter < 10000; counter++)
      {
         System.err.println(counter); // report of the number of bytes so far
         stdinOfChild.print("a");
         //stdinOfChild.flush(); // Why does this change the number of bytes needed to block??
      }
      stdinOfChild.close();
   }
}

/*
   If you comment out the line stdinOfChild.flush() then
   it seems that the parent has twice as much buffer space
   to fill before it blocks. The reason is that it is now
   filling both its own output buffer and the buffer in
   the pipe connecting it to its child, so there really is
   (approximately) twice as much buffer space. When you flush
   the parent's output buffer, each byte moves to the pipe's
   buffer. When the pipe's buffer is finally full, one more
   byte flushed to the pipe causes the parent to block (because
   the child has not yet woken up and started emptying the pipe).
   But that means the whole parent's output buffer is left empty,
   while the parent is blocked on trying to flush one byte to the
   pipe.

   Notice however that if you do not flush() the parent's output
   buffer, the program hangs, since nothing seems to move from the
   pipe's buffer to the child's input buffer.
*/